{ *************************************************************************** }
{                                                                             }
{ Kylix and Delphi Cross-Platform Visual Component Library                    }
{                                                                             }
{ Copyright (c) 1995, 2005 Borland Software Corporation                       }
{                                                                             }
{ *************************************************************************** }


unit FMTBcd;

interface

uses SysUtils, Variants;

const

  MaxStringDigits = 100;
  _NoDecimal = -255;
  _DefaultDecimals = 10;

  { From DB.pas }
  { Max supported by Midas }
  MaxFMTBcdFractionSize = 64;
  { Max supported by Midas }
  MaxFMTBcdDigits =   32;
  DefaultFMTBcdScale = 6;
  MaxBcdPrecision =   18;
  MaxBcdScale     =   4;

type

  PBcd = ^TBcd;
  TBcd  = packed record
    Precision: Byte;                        { 1..64 }
    SignSpecialPlaces: Byte;                { Sign:1, Special:1, Places:6 }
    Fraction: packed array [0..31] of Byte; { BCD Nibbles, 00..99 per Byte, high Nibble 1st }
  end;
  TNibbleArray = array[0..63] of Byte;


{ Exception classes }

  EBcdException = class(Exception);
  EBcdOverflowException = class(EBcdException);

{ Utility functions for TBcd access }

function BcdPrecision(const Bcd: TBcd): Word;
function BcdScale(const Bcd: TBcd): Word;
function IsBcdNegative(const Bcd: TBcd): Boolean;

{ Bcd Arithmetic}

procedure BcdAdd(const bcdIn1, bcdIn2: TBcd; var bcdOut: TBcd); deprecated;
procedure BcdSubtract(const bcdIn1, bcdIn2: TBcd; var bcdOut: TBcd); deprecated;
{ Returns True if successful, False if Int Digits needed to be truncated }
function NormalizeBcd(const InBcd: TBcd; var OutBcd: TBcd; const Prec, Scale: Word): Boolean;

procedure BcdMultiply(const bcdIn1, bcdIn2: TBcd; var bcdOut: TBcd); overload; deprecated;
procedure BcdMultiply(const bcdIn: TBcd; const DoubleIn: Double; var bcdOut: TBcd); overload; deprecated;
procedure BcdMultiply(const bcdIn: TBcd; const StringIn: string; var bcdOut: TBcd); overload; deprecated;
procedure BcdMultiply(StringIn1, StringIn2: string; var bcdOut: TBcd); overload; deprecated;
procedure BcdDivide(Dividend, Divisor: string; var bcdOut: TBcd); overload; deprecated;
procedure BcdDivide(const Dividend, Divisor: TBcd; var bcdOut: TBcd); overload; deprecated;
procedure BcdDivide(const Dividend: TBcd; const Divisor: Double; var bcdOut: TBcd); overload; deprecated;
procedure BcdDivide(const Dividend: TBcd; const Divisor: string; var bcdOut: TBcd); overload; deprecated;

{ TBcd variant creation utils }
procedure VarFMTBcdCreate(var ADest: Variant; const ABcd: TBcd); overload;
function VarFMTBcdCreate: Variant; overload;
function VarFMTBcdCreate(const AValue: string; Precision, Scale: Word): Variant; overload;
function VarFMTBcdCreate(const AValue: Double; Precision: Word = 18; Scale: Word = 4): Variant; overload;
function VarFMTBcdCreate(const ABcd: TBcd): Variant; overload;
function VarIsFMTBcd(const AValue: Variant): Boolean; overload;
function VarFMTBcd: TVarType;

{ Convert String/Double/Integer to BCD struct }
function StrToBcd(const AValue: string): TBcd; 
function TryStrToBcd(const AValue: string; var Bcd: TBcd): Boolean;
function DoubleToBcd(const AValue: Double): TBcd; overload;
procedure DoubleToBcd(const AValue: Double; var bcd: TBcd); overload;
function IntegerToBcd(const AValue: Integer): TBcd;
function VarToBcd(const AValue: Variant): TBcd;

{ From DB.pas }
function CurrToBCD(const Curr: Currency; var BCD: TBcd; Precision: Integer = 32;
  Decimals: Integer = 4): Boolean;
function OldCurrToBCD(const Curr: Currency; var BCD: TBcd; Precision: Integer = 32;
  Decimals: Integer = 4): Boolean;

{ Convert Bcd struct to string/Double/Integer }
function BcdToStr(const Bcd: TBcd): string; overload;
function BcdToDouble(const Bcd: TBcd): Double;
function BcdToInteger(const Bcd: TBcd; Truncate: Boolean = False): Integer;

{ From DB.pas }
function OldBCDToCurr(const BCD: TBcd; var Curr: Currency): Boolean;
function BCDToCurr(const BCD: TBcd; var Curr: Currency): Boolean;
{ Formatting Bcd as string }
function BcdToStrF(const Bcd: TBcd; Format: TFloatFormat; const Precision, Digits: Integer): string;
function FormatBcd(const Format: string; Bcd: TBcd): string;
function BcdCompare(const bcd1, bcd2: TBcd): Integer;

function RoundAt(const Value: string; Position: SmallInt): string;

const

  NullBcd: TBcd = (Precision: 0; SignSpecialPlaces: 0; Fraction: (0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0));

implementation

uses VarUtils, SysConst, DBConsts, TypInfo, Math, Classes;


procedure EmptyBcd(var ABcd: TBcd);
begin
   ABcd.Precision := byte(0);
   ABcd.SignSpecialPlaces := byte(0);
end;

{ Digit is the index into Nibbles: Fractioh[Index] = Index div 2 }
procedure PutTwoBcdDigits(const Nibble1, Nibble2: Byte; var Bcd: TBcd; Digit: Integer);
var
  b: Byte;
begin
  b := Nibble1 SHL 4;
  b := b OR (Nibble2 AND 15);
  Bcd.Fraction[Digit div 2] := b;
end;

{ Digit is the index into Nibbles: Fractioh[Index] = Index div 2;
  Byte returned is either left or right Nibble of Fraction Byte }
function GetBcdDigit(const Bcd: TBcd; Digit: Integer): Byte;
begin
  if Digit mod 2 = 0 then
    Result := Byte((Bcd.Fraction[Digit div 2]) SHR 4)
  else
    Result := Byte(Byte((Bcd.Fraction[Digit div 2]) AND 15));
end;


const

DValue: array[-10..20] of Currency = (0, 0, 0, 0, 0, 0, 0,
                                     0.0001, 0.001, 0.01, 0.1, 1,
                                     10,
                                     100,
                                     1000,
                                     10000,
                                     100000,
                                     1000000,
                                     10000000,
                                     100000000,
                                     1000000000,
                                     10000000000,
                                     100000000000,
                                     1000000000000,
                                     10000000000000,
                                     100000000000000,
                                     0, 0, 0, 0, 0);


{ Currency Value of a Byte for a specific digit column }
function PutCurrencyDigit(Value: Byte; Digit: Integer): Currency;
begin
  Result := DValue[Digit] * Value;
end;

type

{ TFMTBcdVariantType }

  TFMTBcdVariantType = class(TPublishableVariantType)
  protected
    function GetInstance(const V: TVarData): TObject; override;
  public
    procedure Clear(var V: TVarData); override;
    procedure Copy(var Dest: TVarData; const Source: TVarData; const Indirect: Boolean); override;
    procedure Cast(var Dest: TVarData; const Source: TVarData); override;
    procedure CastTo(var Dest: TVarData; const Source: TVarData; const AVarType: TVarType); override;
    procedure BinaryOp(var Left: TVarData; const Right: TVarData; const Operator: TVarOp); override;
    procedure Compare(const Left, Right: TVarData; var Relationship: TVarCompareResult); override;
  end;

var

{ FMTBcd that the complex variant points to }

  FMTBcdVariantType: TFMTBcdVariantType = nil;

type

{ TFMTBcdData }

  TFMTBcdData = class(TPersistent)
  private
    FBcd: TBcd;
    function GetAsCurrency: Currency;
    function GetAsDouble: Double;
    function GetAsInteger: Integer;
    function GetAsString: string;
    function GetAsSmallInt: SmallInt;
    procedure SetAsCurrency(const Value: Currency);
    procedure SetAsDouble(const Value: Double);
    procedure SetAsInteger(const Value: Integer);
    procedure SetAsSmallInt(const Value: SmallInt);
    procedure SetAsString(const Value: string);
  public
    constructor Create(const AValue: Integer); overload;
    constructor Create(const AValue: Double; Precision, Scale: Word); overload;
    constructor Create(const AValue: Currency); overload;
    constructor Create(const AText: string; Precision, Scale: Word); overload;
    constructor Create(const ABcd: TBcd); overload;
    constructor Create(const ASource: TFMTBcdData); overload;

    property Bcd: TBcd read FBcd write FBcd;

    function Compare(const Value: TFMTBcdData): TVarCompareResult;

    procedure DoAdd(const Value: TBcd); overload;
    procedure DoAdd(const AFMTBcd: TFMTBcdData); overload;
    procedure DoSubtract(const Value: TBcd); overload;
    procedure DoSubtract(const AFMTBcd: TFMTBcdData); overload;
    procedure DoMultiply(const ABcdData: TFMTBcdData); overload;
    procedure DoDivide(const ABcdData: TFMTBcdData); overload;
  published
    { Conversion }
    property AsCurrency: Currency read GetAsCurrency write SetAsCurrency;
    property AsDouble: Double read GetAsDouble write SetAsDouble;
    property AsInteger: Integer read GetAsInteger write SetAsInteger;
    property AsSmallInt: SmallInt read GetAsSmallInt write SetAsSmallInt;
    property AsString: string read GetAsString write SetAsString;
  end;

{ Helper record that helps crack open TFMTBcdObject }

  TFMTBcdVarData = packed record
    VType: TVarType;
    Reserved1, Reserved2, Reserved3: Word;
    VBcd: TFMTBcdData;
    Reserved4: LongWord;
  end;

procedure BcdErrorFmt(const Message, BcdAsString: string);
begin
  raise EBcdException.Create(Format(Message, [BcdAsString]));
end;

procedure BcdError(const Message: string);
begin
  raise EBcdException.Create(Message);
end;

procedure OverflowError(const Message: string);
begin
  raise EBcdOverflowException.Create(Message);
end;

{ TFMTBcdData }

procedure TFMTBcdData.DoAdd(const Value: TBcd);
var
  NewBcd: TBcd;
begin
  EmptyBcd(NewBcd);
  BcdAdd(Self.Bcd, Value, NewBcd);
  Self.Bcd := NewBcd;
end;

procedure TFMTBcdData.DoAdd(const AFMTBcd: TFMTBcdData);
begin
  DoAdd(AFMTBcd.Bcd);
end;

procedure TFMTBcdData.DoSubtract(const Value: TBcd);
var
  NewBcd: TBcd;
begin
  EmptyBcd(NewBcd);
  BcdSubtract(Self.Bcd, Value, NewBcd);
  Self.Bcd := NewBcd;
end;

procedure TFMTBcdData.DoSubtract(const AFMTBcd: TFMTBcdData);
begin
  DoSubtract(AFMTBcd.Bcd);
end;

procedure TFMTBcdData.DoMultiply(const ABcdData: TFMTBcdData);
var
  ABcd: TBcd;
begin
  EmptyBcd(ABcd);
  BcdMultiply(Self.Bcd, ABcdData.Bcd, ABcd);
  Self.Bcd := ABcd;
end;

procedure TFMTBcdData.DoDivide(const ABcdData: TFMTBcdData);
var
  ABcd: TBcd;
begin
  EmptyBcd(ABcd);
  BcdDivide(Self.Bcd, ABcdData.Bcd, ABcd);
  Self.Bcd := ABcd;
end;

function TFMTBcdData.Compare(const Value: TFMTBcdData): TVarCompareResult;
begin
  Result := TVarCompareResult(BcdCompare(Self.FBcd, Value.FBcd)+1);
end;

function TFMTBcdData.GetAsString: string;
begin
  Result := BcdToStr(Self.FBcd);
end;

function TFMTBcdData.GetAsDouble: Double;
begin
  Result := StrToFloat(BcdToStr(Self.FBcd));
end;

function TFMTBcdData.GetAsInteger: Integer;
begin
  Result := StrToInt(BcdToStr(Self.FBcd));
end;

function TFMTBcdData.GetAsSmallInt: SmallInt;
begin
  Result := SmallInt(GetAsInteger);
end;

function TFMTBcdData.GetAsCurrency: Currency;
begin
  BCDToCurr(Self.FBcd, Result);
end;

procedure TFMTBcdData.SetAsString(const Value: string);
begin
  FBcd := StrToBcd(Value);
end;

procedure TFMTBcdData.SetAsDouble(const Value: Double);
begin
  FBcd := StrToBcd(FloatToStr(Value));
end;

procedure TFMTBcdData.SetAsInteger(const Value: Integer);
begin
  FBcd := StrToBcd(IntToStr(Value));
end;

procedure TFMTBcdData.SetAsSmallInt(const Value: SmallInt);
begin
  SetAsInteger(Integer(Value));
end;

procedure TFMTBcdData.SetAsCurrency(const Value: Currency);
begin
  CurrToBcd(Value, FBcd);
end;

constructor TFMTBcdData.Create(const ABcd: TBcd);
begin
  inherited Create;
  Move(ABcd, FBcd, SizeOf(TBcd));
end;

constructor TFMTBcdData.Create(const AValue: Integer);
begin
  Create(IntegerToBcd(AValue));
end;

constructor TFMTBcdData.Create(const AValue: Double; Precision, Scale: Word);
var
  ABcd, OutBcd: TBcd;
begin
  if (Precision = 0) or (Precision > MaxFMTBcdFractionSize) or (Scale > Precision) then
    OverflowError(SBcdOverflow);
  ABcd := StrToBcd(FloatToStr(AValue));
  if not NormalizeBcd(ABcd, OutBcd, Precision, Scale) then
    OverflowError(SBcdOverflow);
  Create(OutBcd);
end;

constructor TFMTBcdData.Create(const AValue: Currency);
var
  OutBcd: TBcd;
begin
  CurrToBcd(AValue, OutBcd);
  Create(OutBcd);
end;

constructor TFMTBcdData.Create(const AText: string; Precision, Scale: Word);
var
  ABcd, OutBcd: TBcd;
begin
  ABcd := StrToBcd(AText);
  if not NormalizeBcd(ABcd, OutBcd, Precision, Scale) then
    OverflowError(SBcdOverflow);
  Create(OutBcd);
end;

constructor TFMTBcdData.Create(const ASource: TFMTBcdData);
begin
  Create(aSource.Bcd);
end;

{ TFMTBcdVariantType }

procedure TFMTBcdVariantType.Clear(var V: TVarData);
begin
  V.VType := varEmpty;
  FreeAndNil(TFMTBcdVarData(V).VBcd);
end;

procedure TFMTBcdVariantType.Cast(var Dest: TVarData;
  const Source: TVarData);
var
  LSource, LTemp: TVarData;
begin
  VarDataInit(LSource);
  try
    VarDataCopyNoInd(LSource, Source);
    if VarDataIsStr(LSource) then
      TFMTBcdVarData(Dest).VBcd := TFMTBcdData.Create(VarDataToStr(LSource), 32, 8)
    else
    begin
      VarDataInit(LTemp);
      try
        VarDataCastTo(LTemp, LSource, varDouble);
        TFMTBcdVarData(Dest).VBcd := TFMTBcdData.Create(LTemp.VDouble, 32, 8);
      finally
        VarDataClear(LTemp);
      end;
    end;
    Dest.VType := VarType;
  finally
    VarDataClear(LSource);
  end;
end;

procedure TFMTBcdVariantType.CastTo(var Dest: TVarData;
  const Source: TVarData; const AVarType: TVarType);
var
  LTemp: TVarData;
begin
  if Source.VType = VarType then
    case AVarType of
      varOleStr:
        VarDataFromOleStr(Dest, TFMTBcdVarData(Source).VBcd.AsString);
      varString:
        VarDataFromStr(Dest, TFMTBcdVarData(Source).VBcd.AsString);
    else
      VarDataInit(LTemp);
      try
        LTemp.VType := varDouble;
        LTemp.VDouble := BcdToDouble(TFMTBcdVarData(Source).VBcd.Bcd);
        VarDataCastTo(Dest, LTemp, AVarType);
      finally
        VarDataClear(LTemp);
      end;
    end
  else
    inherited;
end;

procedure TFMTBcdVariantType.Copy(var Dest: TVarData; const Source: TVarData; const Indirect: Boolean);
begin
  if Indirect and VarDataIsByRef(Source) then
    VarDataCopyNoInd(Dest, Source)
  else
    with TFMTBcdVarData(Dest) do
    begin
      VType := VarType;
      VBcd := TFMTBcdData.Create(TFMTBcdVarData(Source).VBcd);
    end;
end;

procedure TFMTBcdVariantType.BinaryOp(var Left: TVarData; const Right: TVarData; const Operator: TVarOp);
begin
  case Operator of
    opAdd:
      TFMTBcdVarData(Left).VBcd.DoAdd(TFMTBcdVarData(Right).VBcd);
    opSubtract:
      TFMTBcdVarData(Left).VBcd.DoSubtract(TFMTBcdVarData(Right).VBcd);
    opMultiply:
      TFMTBcdVarData(Left).VBcd.DoMultiply(TFMTBcdVarData(Right).VBcd);
    opDivide:
      TFMTBcdVarData(Left).VBcd.DoDivide(TFMTBcdVarData(Right).VBcd);
  else
    RaiseInvalidOp;
  end;
end;

function TFMTBcdVariantType.GetInstance(const V: TVarData): TObject;
begin
  Result := TFMTBcdVarData(V).VBcd;
end;

procedure TFMTBcdVariantType.Compare(const Left, Right: TVarData; var Relationship: TVarCompareResult);
begin
  Relationship := TFMTBcdVarData(Left).VBcd.Compare(TFMTBcdVarData(Right).VBcd);
end;

{ FMTBcd variant create utils }

function VarFMTBcdCreate(const AValue: string; Precision, Scale: Word): Variant; overload;
begin
  VarClear(Result);
  TFMTBcdVarData(Result).VType := FMTBcdVariantType.VarType;
  TFMTBcdVarData(Result).VBcd := TFMTBcdData.Create(AValue, Precision, Scale);
end;

function VarFMTBcdCreate(const AValue: Double; Precision, Scale: Word): Variant; overload;
begin
  VarClear(Result);
  TFMTBcdVarData(Result).VType := FMTBcdVariantType.VarType;
  TFMTBcdVarData(Result).VBcd := TFMTBcdData.Create(AValue, Precision, Scale);
end;

procedure VarFMTBcdCreate(var ADest: Variant; const ABcd: TBcd); overload;
begin
  VarClear(ADest);
  TFMTBcdVarData(ADest).VType := FMTBcdVariantType.VarType;
  TFMTBcdVarData(ADest).VBcd := TFMTBcdData.Create(ABcd);
end;

function VarFMTBcdCreate: Variant; overload;
begin
  VarFMTBcdCreate(Result, NullBcd);
end;

function VarFMTBcdCreate(const ABcd: TBcd): Variant;
begin
  VarFMTBcdCreate(Result, ABcd);
end;

function VarIsFMTBcd(const AValue: Variant): Boolean;
begin
  Result := (TVarData(AValue).VType = FMTBcdVariantType.VarType);
end;

function VarFMTBcd: TVarType;
begin
  Result := FMTBcdVariantType.VarType;
end;

function StrToBcd(const AValue: string): TBcd;
var
  Success: Boolean;
begin
  Success := TryStrToBcd(AValue, Result);
  if not Success then
    BcdErrorFmt(SInvalidBcdValue, AValue);
end;

procedure DoubleToBcd(const AValue: Double; var bcd: TBcd); overload;
begin
  bcd := StrToBcd(FloatToStr(AValue));
end;
function DoubleToBcd(const AValue: Double): TBcd; overload;
begin
  DoubleToBcd(AValue, Result);
end;

function VarToBcd(const AValue: Variant): TBcd;
begin
  if VarType(AValue) = FMTBcdVariantType.VarType then
    Result := TFMTBcdVarData(AValue).VBcd.FBcd
  else
    Result := TFMTBcdVarData(VarFmtBcdCreate(AValue)).VBcd.FBcd;
end;

function IntegerToBcd( const AValue: Integer): TBcd;
begin
  Result := StrToBcd(IntToStr(AValue));
end;

function BcdToDouble(const Bcd: TBcd): Double;
begin
  Result := StrToFloat(BcdToStr(Bcd));
end;

function BcdToInteger(const Bcd: TBcd; Truncate: Boolean = False): Integer;
var
  ABcd: TBcd;
begin
  if (Truncate) and (BcdScale(Bcd) > 0 ) then
    NormalizeBcd(Bcd, ABcd, Bcd.Precision, 0)
  else
    ABcd := Bcd;
  Result := StrToInt(BcdToStr(ABcd));    
end;

{ utility routines }

{ used internally, and only to round decimals, not integer portion }
{ requires at least 1 digit!! }
function RoundAt(const Value: string; Position: SmallInt): string;

  Procedure RoundChar(const PrevChar: SmallInt; var Carry: Boolean);
  begin
    if Result[PrevChar] in ['0' .. '9'] then
    begin
      if Result[PrevChar] = '9' then
      begin
        Result[PrevChar] := '0';
        Carry := True;
      end else
      begin
        Result[PrevChar] := Char(Byte(Result[PrevChar]) + 1);
        Carry := False;
      end;
    end;
  end;

var
  C, Dot: Char;
  PrevChar, I, DecPos, DecDigits: SmallInt;
  Carry: Boolean;
  Neg: string;
begin
  Dot := DecimalSeparator;
  if Value[1] = '-' then
  begin
    Result := Copy(Value, 2, MaxInt);
    Neg := '-';
  end else
  begin
    Result := Value;
    Neg := '';
  end;
  DecPos := Pos(Dot, Result);
  if DecPos > 0 then
    DecDigits := Length(Result) - DecPos
  else
    DecDigits := 0;
  if (DecPos = 0) or (DecDigits <= Position) then
    { nothing to round }
  begin
    Result := Value;
    Exit;
  end;
  if Result[DecPos + Position + 1] < '5' then
  begin
    { no possible rounding required }
    if Position = 0 then
      Result := Neg + Copy(Result, 1, DecPos + Position -1)
    else
      Result := Neg + Copy(Result, 1, DecPos + Position);
  end else
  begin
    Carry := False;
    PrevChar := 1;
    for I := DecPos + DecDigits downto (DecPos + 1 + Position) do
    begin
      C := Result[I];
      PrevChar := I-1;
      if Result[PrevChar] = Dot then
      begin
        Dec(PrevChar);
        Dec(Position);
      end;
      if (Byte(C) >= 53) or Carry then { if '5' or greater }
        RoundChar(PrevChar, Carry);
    end;
    while Carry do
    begin
      if PrevChar >= DecPos then
        Dec(Position);
      Dec(PrevChar);
      if PrevChar = 0 then
        break;
      if Result[PrevChar] <> Dot then
        RoundChar(PrevChar, Carry);
    end;
    if Carry then
      Result := Neg + '1' + Copy(Result, 1, DecPos + Position)
    else
      Result := Neg + Copy(Result, 1, DecPos + Position);
  end;    
end;

function LeftTrim(const Value: string): string;
begin
  Result := Value;
  while (Length(Result) > 1) and (Result[1] = '0') do
    Result := Copy(Result, 2, Length(Result) -1);
end;

function CompareDigits(S1, S2: string): Integer;
begin
  S1 := LeftTrim(S1);
  if Length(S1) > Length(S2) then
    Result := 1
  else if Length(S2) > Length(S1) then
    Result := -1
  else
    Result := CompareStr(S1, S2);
end;

procedure GetValueAndMultiplyOrder(A, B: string; var V, M: string; LA, LB: Integer; var Wid, Len, DecPos: Integer);
var
  DecimalPosA, DecimalPosB: Integer;
  Dot : Char;
begin
  Dot := DecimalSeparator;
  DecPos := 0;
  if CompareDigits(A,B) > 1 then
  begin
    V := A;
    M := B;
    Wid := LA;
    Len := LB;
  end else
  begin
    M := A;
    V := B;
    Wid := LB;
    Len := LA;
  end;  
  { to get rid of GetDecimalPosition }
  DecimalPosA := Pos(Dot, V);
  DecimalPosB := Pos(Dot, M);
  if (DecimalPosA = 0) and (DecimalPosB = 0) then
    DecPos := _NoDecimal
  else
  begin
    if DecimalPosA > 0 then
    begin
      V := StringReplace(V, Dot, '', []);
      DecPos := Wid - DecimalPosA;
      Dec(Wid);
    end;
    if DecimalPosB > 0 then
    begin
      M := StringReplace(M, Dot,'',[]);
      DecPos := DecPos + (Len - DecimalPosB);
      Dec(Len);
    end;
  end;
end;

function AddChars(V, M: Char; var R: Byte): string;
var
  Value: Byte;
begin
  Value := Byte((Byte(V)-48) + (Byte(M)-48) + R);
  if Value > 9 then
  begin
    Result := Char((Value - 10) + 48);
    R := Byte(1);
  end else
  begin
    Result :=  Char(Value + 48);
    R := Byte(0);
  end;
  if Result = '' then Result := '0';
end;

function SubtractChars(V, M: Char; var R: Byte): string;
var
  Value: Byte;
begin
  Value := Byte((Byte(V)-48) - ((Byte(M)-48) + R));
  if Value > 9 then  // Byte is unsigned: values will be between 246-255
  begin
    Result := Char((Value + 10) + 48);
    R := Byte(1);
  end else
  begin
    Result :=  Char(Value + 48);
    R := Byte(0);
  end;
end;

function AddStrings(const V, M: string): string;
var
  Digit: string;
  LenV, LenM, MaxDigits, I, DigitV, DigitM: Integer;
  R: Byte;           // Remainder
  CV, CM: Char;      // char from string A, string B
begin
  if (V = '') or (V = '0') then
  begin
    Result := M;
    Exit;
  end;
  if (M = '') or (M = '0') then
  begin
    Result := V;
    Exit;
  end;
  R := 0;
  LenV := Length(V);
  LenM := Length(M);
  MaxDigits := Max(LenV, LenM);
  Result := '';
  for I := 0 to MaxDigits - 1 do
  begin
    DigitV := LenV - I;
    DigitM := LenM - I;
    if I < LenV then CV := V[DigitV] else CV := '0';
    if I < LenM then CM := M[DigitM] else CM := '0';
    Digit := AddChars(CV, CM, R);
    Result := Digit + Result;
  end;
  if R > 0 then
    Result := '1' + Result;
end;

function SubtractStrings(const Value, Minus: string): string;
var
  Digit, V, M: string;
  LenV, LenM, MaxDigits, I, DigitV, DigitM: Integer;
  R: Byte;           // Remainder
  CV, CM: Char;      // char from string A, string B
begin
  if CompareDigits(Value, Minus) >= 0 then
  begin
    V := Value;
    M := Minus;
    Result := '';
  end else
  begin
    M := Value;
    V := Minus;
    Result := '-';
  end;
  if (V = '') or (M = '') then
  begin
    if V = '' then
      Result := '-' + M
    else
      Result := V;
  end;
  if (V = '0') or (M = '0') then
  begin
    if M = '0' then
      Result := V
    else if V = '0' then
      Result := '0'
    else
      Result := M;
    Exit;
  end;
  R := 0;
  LenV := Length(V);
  LenM := Length(M);
  MaxDigits := Max(LenV, LenM);
  Result := '';
  for I := 0 to MaxDigits - 1 do
  begin
    DigitV := LenV - I;
    DigitM := LenM - I;
    if I < LenV then
      CV := V[DigitV]
    else
      CV := '0';
    if I < LenM then
      CM := M[DigitM]
    else
      CM := '0';
    Digit := SubtractChars(CV, CM, R);
    Result := Digit + Result;
  end;
  if Result = '' then
    Result := '0';
end;

function _ShortMultiply(const A, B: string): string;
var
  DecPos,W,L,I: Integer;
  S, SS, Times: LongWord;
  Value, Multiplier: string;
begin
  GetValueAndMultiplyOrder(A, B, Value, Multiplier, Length(A), Length(B), W, L, DecPos);
  Times := 1;
  S := 0;
  for I := L downto 1 do
  begin
    SS := LongWord(LongWord((Ord(Multiplier[I])-Byte(48))) * LongWord(StrToInt(Value)) * Times);
    S := SS + S;
    Times := Times * 10;
  end;
  Result := IntToStr(S);
  if DecPos <> _NoDecimal then
  begin
    I := Length(Result) - DecPos;
    if I = 0 then
      Result := '0' + DecimalSeparator + Result
    else if I > 0 then
      Result := Copy(Result,1, I) + DecimalSeparator + Copy(Result,I+1,DecPos)
    else if I < 0 then
      Result := DecimalSeparator + StringOfChar('0', -I) + Result;
  end;
end;

function StringMultiplyByByte(const Value: string; B: Byte): string; pascal;
var
  I: SmallInt;
  R, SB: Byte;
begin
  R := 0;
  Result := '';
  for I := Length(Value) downto 1 do
  begin
    SB :=  ((Byte(Value[I]) - 48) *B) + R;
    R := SB Div 10;
    if R > 0 then
      SB := SB mod 10;
    Result := Char(48 + SB) + Result;
  end;
  if R > 0 then
    Result := Char(48 + R) + Result
end;

function _LongMultiply(const A, B: string): string;
const
  MaxSmall = 9;
var
  DecPos, W, L, I: Integer;
  Times, S, SS: string;
  Value, Multiplier: string;
  T1, T2: Integer;
begin
  GetValueAndMultiplyOrder(A, B, Value, Multiplier, Length(A), Length(B), W, L, DecPos);
  T1 := Length(Value);
  T2 := Length(Multiplier);
  Times := '';
  S := '';
  for I := L downto 1 do
  begin
    if W < MaxSmall then
      SS := IntToStr(LongWord((Ord(Multiplier[I])-48) * StrToInt(Value))) + Times
    else
      SS := StringMultiplyByByte(Value, Ord(Multiplier[I])-48) + Times;
    S := AddStrings(SS,S);
    Times := Times + '0';
  end;
  Result := LeftTrim(S);
  if DecPos <> _NoDecimal then
  begin
    I := Length(Result) - DecPos;
    if I = 0 then
      Result := '0' + DecimalSeparator + Result
    else if I > 0 then
      Result := Copy(Result,1, I) + DecimalSeparator + Copy(Result,I+1,DecPos)
    else if I < 0 then
      Result := DecimalSeparator + StringOfChar('0', -I) + Result;
    if T1 + T2 > 1024 then Result := ''
  end;
end;

{ Go through 2 strings and determine if total length > MaxDigits }
function CanUseShort(A, B: string; MaxDigits: ShortInt): Boolean; 
var
  Len, I: ShortInt;
  Dot: String;
begin
  Len := Length(A) + Length(B);
  Result := Len <= MaxDigits;
  if not Result then
  begin
    Dot := DecimalSeparator;
    if Pos(Dot, A) > 0 then
      Dec(Len, Length(Dot));
    if Pos(Dot, B) > 0 then
      Dec(Len, Length(Dot));
    Result := Len <= MaxDigits;
    if not Result then
    begin
      I := 1;
      while (Len > 0) and (A[I] = '0') and (I <= Length(A)) do
      begin
        Inc(I);
        Dec(Len);
      end;
      I := 1;
      while (Len > 0) and (B[I] = '0') and (I <= Length(B)) do
      begin
        Inc(I);
        Dec(Len);
      end;
      Result := Len <= MaxDigits;
    end;
  end;
end;

// Return True if S evaluates to 0
function BlankArgument(const S: string): Boolean;
var
  I: Integer;
  Dot: Char;
begin
  Dot := DecimalSeparator;
  Result := True;
  for I := 1 to Length(S) do
    if not (S[I] in ['0', Dot]) then
    begin
      Result := False;
      break;
    end;
end;

function _Multiply(const A, B: string): string;
begin
  if BlankArgument(A) or BlankArgument(B) then
    Result := '0'
  else if CanUseShort(A, B, 9) then
    Result := _ShortMultiply(A,B)
  else
    Result := _LongMultiply(A,B);
end;

function NextDigit(const V, D: string; var R: string): string;
begin
  R := V;
  Result := '0';
  while CompareDigits(R, D) >= 0 do
  begin
    Result := IntToStr(StrToInt(Result) + 1);
    R := LeftTrim(SubtractStrings(R, D));
  end;
end;

function AdjustDecimalPosition(const Value: string; DecPos: SmallInt): string;
var
  Dot : Char;
begin
  Dot := DecimalSeparator;
  Result := Value;
  while Result[1] = '0' do 
    Result := Copy(Result, 2, length(Result) -1);
  if DecPos = 0 then
    Result := '0' + Dot + Result
  else if DecPos > 0 then
    Result := '0' + Dot + StringOfChar('0', DecPos) + Result
  else // DecPos < 0 then
  begin
    if -DecPos >= Length(Result) then
      Result := Result + StringOfChar('0', (-DecPos)-Length(Result))
    else if -DecPos < Length(Result) then
    begin
      Result := Copy(Result, 1, -DecPos) + Dot + Copy(Result, (-DecPos)+1, Length(Result));
    end;
  end;
end;

function ValueOverOne(D: string): string;
var
  R: string;
  V: string;
  AddZeros, DecimalPos: SmallInt;
  Dot : Char;
begin
  Dot := DecimalSeparator;
  DecimalPos := Pos(Dot, D);
  if DecimalPos > 0 then
  begin
    Result := '10';
    Dec(DecimalPos,2);  // 1/.2 = 5.0; 1/2.2 = .45;
    if DecimalPos = -1 then // D[1] is DecimalSeparator
    begin
      D := Copy(D, 2, Length(D) -1);
      while D[1] = '0' do
      begin
        Result := Result + '0';       // copy back later
        D := Copy(D, 2, Length(D) -1);
        Dec(DecimalPos);
      end;
    end else
      D := StringReplace(D, Dot, '', []);
  end else
  begin
    DecimalPos := Length(D) -1;
    Result := '1';
  end;
  if (D ='1') or (D = '1' + StringOfChar('0', Length(D) -1)) then
    Result := AdjustDecimalPosition(Result, DecimalPos -1)
  else
  begin
    V := '1';
    R := V;
    AddZeros := Length(V) -1;  // for divisor of 12345, add 4 zeros
    V := V + StringOfChar('0', AddZeros);
    if CompareDigits(V,D) < 0 then   // if still less add 1
      V := V + '0';
    Result := '';
    while (R <> '0') and (Length(Result) < (MaxFMTBcdFractionSize + AddZeros)) do
    begin
      Result := Result + NextDigit(V, D, R);
      V := R + '0';
    end;
    Result := AdjustDecimalPosition(Result, DecimalPos);
  end;
end;

function _LongDivide(A, B: string): string;
var
  Negative: Boolean;
  FinalDecimalPos, Decimals : Integer;
  Dot: Char;
begin
  Dot := DecimalSeparator;
  Result := '0';
  // save pos/minus info and remove '-'
  Negative := (A[1] <> B[1]) and ((A[1] = '-') or (B[1] = '-'));
  if A[1] = '-' then A := Copy(A, 2, Length(A)-1);
  if B[1] = '-' then B := Copy(B, 2, Length(B)-1);
  if A = '0' then
    Exit;
  while A[1] = '0' do A := Copy(A, 2, Length(A)-1);
  while B[1] = '0' do B := Copy(B, 2, Length(B)-1);
  Result := ValueOverOne(B);
  Result := _Multiply(A, Result);
  Decimals := Length(A) - Pos(Dot, A);
  if Length(B) - Pos(Dot, B) > Decimals then
    Decimals := Length(B) - Pos(Dot, B);
  FinalDecimalPos := Pos(Dot, Result);
  { if there are more than DecimalPos + 20 decimals, truncate }
  if (Length(Result) - FinalDecimalPos) > (Decimals + 20) then
    SetLength(Result, FinalDecimalPos + Decimals + 20);
  if Negative then
    Result := '-' + Result;
end;

function _Divide(const A, B: string): string;
begin
  if BlankArgument(A) and BlankArgument(B) then
  begin
    if (A = '') or (A = '0') then
      Result := '0'
    else
      BcdError(SDivByZero);
  end
  else if B = '1' then
    Result := A
  else if B = '-1' then
    Result := '-' + A
  else if CompareStr(A,B) = 0 then
    Result := '1'
  else
    Result := _LongDivide(A,B);
end;

function InvalidBcdString(Value: string): Boolean;
var
  I, DecPos, DecEnd: Integer;
  Dot: Char;
begin
  Dot := DecimalSeparator;
  Result := False;
  DecPos := Pos(Dot, Value);
  if DecPos = 0 then
    DecEnd := DecPos
  else
    DecEnd := DecPos + Length(Dot);
  for I := 1 to Length(Value) do
    if (not (Value[I] in ['0'..'9', '-', '+'])) and
       ((I < DecPos) or (I >= DecEnd))  then
    begin
      Result := True;
      break;
    end;
end;

function ReverseNegative(SignSpecialPlaces: Byte): Byte;
begin
  if (SignSpecialPlaces and (1 shl 7)) <> 0 then
    Result := (SignSpecialPlaces and 63)
  else
    Result := (SignSpecialPlaces and 63) or (1 shl 7);
end;

procedure ZeroBcd(var Bcd: TBcd; FractionOnly: Boolean = False);
var
 I: Integer;
begin
  if not FractionOnly then
  begin
    Bcd.Precision := 10;
    Bcd.SignSpecialPlaces := 2;
  end;
  for I := 0 to 31 do
    Bcd.Fraction[I] := 0;
end;

function TryStrToBcd(const AValue: string; var Bcd: TBcd): Boolean;
const
  spaceChars = [ ' ', #6, #10, #13, #14];
var
  Neg: Boolean;
  ICopyDigits, NumDigits: Word;
  DecimalPos: SmallInt;
  Source: string;
  Nibble1, Nibble2: Byte;
  Dot: Char;
begin
  Dot := DecimalSeparator;
  ZeroBcd(Bcd);
  if InvalidBcdString(AValue) then
  begin
    Result := False;
    exit;
  end;
  if (AValue = '0') or (AValue = '') then
  begin
    Result := True;
    Exit;
  end;
  Result := True;
  Neg := False;
  DecimalPos := Pos(Dot, AValue);
  if DecimalPos = 0 then
    DecimalPos := -1;

  NumDigits := Length(AValue);

  { Strip leading whitespace }
  iCopyDigits := 1;
  while (iCopyDigits <= NumDigits) and
    (AValue[iCopyDigits] in spaceChars) or (AValue[iCopyDigits] in ['0', '+', '-']) do
  begin
    if AValue[iCopyDigits] = '-' then
      Neg := True;
    Inc(iCopyDigits);
    if DecimalPos > 0 then
      Dec(DecimalPos);
  end;
  NumDigits := NumDigits + 1 - ICopyDigits;
  Source := Copy(AValue, iCopyDigits, NumDigits);
  if (NumDigits = 0) or (AValue = '0') or (AValue = '') then
  begin
    Exit;
  end;

 { Strip trailing whitespace }
  ICopyDigits := NumDigits;
  while (Source[ICopyDigits] in spaceChars) do
    Dec(ICopyDigits);
  if ICopyDigits <> NumDigits then
  begin
    SetLength(Source, ICopyDigits);
    NumDigits := ICopyDigits;
  end;

  if (NumDigits > MaxFMTBcdFractionSize) then
  begin
    if (DecimalPos >= 0) and (DecimalPos <= MaxFMTBcdFractionSize) then
      NumDigits := MaxFMTBcdFractionSize // truncate to 64
    else
      BcdError(sBcdOverflow);
  end;

  if NumDigits > 0 then
  begin
    if DecimalPos > 0 then
    begin
      Source := StringReplace(Source, Dot, '', []);
      Dec(NumDigits);
      Dec(DecimalPos);
    end;
    if Length(Source) mod 2 = 1 then
    begin
      { enforce even # of nibbles }
      Source := '0' + Source;
      Inc(NumDigits);
      if DecimalPos >= 0 then
        Inc(DecimalPos);
    end;
    ICopyDigits := 1;
    while ICopyDigits <= NumDigits do
    begin
      Nibble1 := Byte(Ord(Source[ICopyDigits]) - 48);
      Nibble2 := Ord(Source[ICopyDigits+1]) - Ord('0');
      Nibble2 := Nibble2 and $0F; { just to be safe}
      PutTwoBcdDigits(Nibble1, Nibble2, Bcd, ICopyDigits-1);
      Inc(ICopyDigits, 2);
    end;
    Bcd.Precision := NumDigits;
    if DecimalPos >= 0 then
      Bcd.SignSpecialPlaces := NumDigits - DecimalPos
    else
      Bcd.SignSpecialPlaces := 0;
  end
  else begin
    Bcd.Precision := 10;
    Bcd.SignSpecialPlaces := 2;
  end;

  if Neg then
    Bcd.SignSpecialPlaces := (Bcd.SignSpecialPlaces and 63) or (1 shl 7);
end;

function SignificantIntDigits(const BcdIn: TBcd; Digits: Word): Word;
var
  b, Nibble: byte;
  I, J: Integer;
begin
  I :=0;
  Result := Digits;
  while Result > 0 do
  begin
    b := BcdIn.Fraction[I];
    if b = 0 then
    begin
      for J := 1 to 2 do
        if Result > 0 then
          Result := Result -1;
      if Result = 0 then
        break;
      Inc(I);
    end else
    begin
      { check for empty last nibble here: if empy dec Result }
      Nibble := b SHR 4;
      if Nibble = 0 then
        Dec(Result);
      break;
    end;
  end;
end;

procedure AddSubtractNormalizedFractions(const BcdIn1, BcdIn2: TBcd; var BcdOut: TBcd; Subtract: Boolean);var
  TwoNibbles1, TwoNibbles2, Nib1, Nib2, Remainder: Byte;
  SumNib1, SumNib2: ShortInt;
  I : Integer;
begin
  BcdOut.Precision := BcdIn1.Precision;
  if BcdIn1.Precision mod 2 = 0 then
    I := (BcdIn1.Precision div 2) -1
  else  
    I := (BcdIn1.Precision div 2);
  Remainder := 0;
  while I >= 0 do
  begin
    TwoNibbles1 := BcdIn1.Fraction[I];
    TwoNibbles2 := BcdIn2.Fraction[I];
    Nib1 := Byte(Byte(TwoNibbles1 AND 15));
    Nib2 := Byte(Byte(TwoNibbles2 AND 15));
    if Subtract then
    begin
      SumNib1 := Nib1 - (Nib2 + Remainder);
      if SumNib1 < 0 then
      begin
        Remainder := 1;
        Inc(SumNib1, 10);
      end else
        Remainder := 0;
    end else
    begin
      SumNib1 := Nib1 + Nib2 + Remainder;
      Remainder := SumNib1 div 10;
      SumNib1 := SumNib1 mod 10;
    end;

    Nib1 := Byte(TwoNibbles1 SHR 4);
    Nib2 := Byte(TwoNibbles2 SHR 4);
    if Subtract then
    begin
      SumNib2 := Nib1 - (Nib2 + Remainder);
      if SumNib2 < 0 then
      begin
        Remainder := 1;
        Inc(SumNib2, 10);
      end else
        Remainder := 0;
    end else
    begin
      SumNib2 := Nib1 + Nib2 + Remainder;
      Remainder := SumNib2 div 10;
      SumNib2 := SumNib2 mod 10;
    end;
    Nib1 := SumNib2 SHL 4;
    Nib1 := Nib1 OR (Byte(SumNib1) AND 15);
    BcdOut.Fraction[I] := Nib1;
    Dec(I);
  end;
end;


procedure SubtractNormalizedFractions(const BcdIn1, BcdIn2: TBcd; var BcdOut: TBcd);
begin
  AddSubtractNormalizedFractions(BcdIn1, BcdIn2, BcdOut, True);
end;

procedure AddNormalizedFractions(const BcdIn1, BcdIn2: TBcd; var BcdOut: TBcd);
begin
  AddSubtractNormalizedFractions(BcdIn1, BcdIn2, BcdOut, False);
end;

function CompareNormalizedFractions(const BcdIn1, BcdIn2: TBcd; Digits1, Digits2: SmallInt): Integer;
var
  I, Count: SmallInt;
  N1, N2: Byte;
begin
  Count := Min(Digits1, Digits2) div 2;
  for I := 0 to Count - 1 do
    if BcdIn1.Fraction[I] <> BcdIn2.Fraction[I] then
    begin
      // compare first nibble
      N1 := BcdIn1.Fraction[I] shr 4;
      N2 := BcdIn2.Fraction[I] shr 4;
      Result := N1 - N2;
      if Result = 0 then // first nibble was the same, try second
      begin
        N1 := BcdIn1.Fraction[I] and $0f;
        N2 := BcdIn2.Fraction[I] and $0f;
        Result := N1 - N2;
      end;
      Exit;
    end;
  // if we got here, they matched except one has more digits -- check if remaining are 0
  Result := Digits1 - Digits2; // assume they are not all zeros
  if Digits1 > Digits2 then
  begin
    for I := Count to (Digits1 div 2) - 1 do
      if BcdIn1.Fraction[I] <> 0 then
        Exit;
  end else
    for I := Count to (Digits2 div 2) - 1 do
      if BcdIn2.Fraction[I] <> 0 then
        Exit;
  Result := 0; // they were all zeros, change result
end;

procedure MoveBytesToRight(var BcdVal: TBcd; BcdIn: TBcd);
var
  b: Byte;
  I, Last: Integer;
begin
  b := 0;
  Last := (BcdVal.Precision div 2);
  for I := 0 to Last -1 do
  begin
    BcdVal.Fraction[I] := b;
    b := BcdIn.Fraction[I];
  end;
  BcdVal.Fraction[Last] := b;
  if Last mod 2 = 1 then
    BcdVal.Fraction[Last+1] := Byte(B AND 15);
end;

procedure MoveBytesToLeft(var BcdVal: TBcd; BcdIn: TBcd);
var
  b: Byte;
  I, Last: Integer;
begin
  if Byte(BcdIn.Fraction[0]) <> 0 then
    BcdError(sBcdOverflow);
  Last := (BcdVal.Precision div 2);
  b := byte(0);
  for I := 0 to Last -1 do
  begin
    if I > 0 then
      BcdVal.Fraction[I-1] := b;
    b := BcdIn.Fraction[I];
  end;
  if BcdVal.Precision mod 2 = 1 then
    BcdVal.Fraction[Last] := BcdVal.Fraction[Last];
end;

{ Shift Fractions one Nibble to Left }
procedure MoveNibblesToRight(var BcdVal: TBcd; BcdIn: TBcd);
var
  InIndex, OutIndex, InPrecision: Integer;
  OutB, B: Byte;
  nibbleArray: TNibbleArray;
begin
  InPrecision := BcdIn.Precision;
  nibbleArray[0] := 0;
  OutIndex := 1;
  for InIndex := 0 to (InPrecision div 2) do
  begin
    B := Byte(BcdIn.Fraction[InIndex]);
    OutB := nibbleArray[OutIndex-1] SHL 4;
    nibbleArray[OutIndex] := Byte(B SHR 4);
    OutB := OutB OR (nibbleArray[OutIndex] AND 15);
    Inc(OutIndex);
    nibbleArray[OutIndex] := Byte(B AND 15);
    Inc(OutIndex);
    BcdVal.Fraction[InIndex] := OutB;
  end;
  if OutIndex < 32 then
    BcdVal.Fraction[OutIndex] := 0;
end;

// by them time code arrives here, it is known that
// the number of significant integer digits is the same.
function CompareSignificantFractions(const Bcd1, Bcd2: TBcd; IntDigits: Word): Integer;

  function LoadNibbleArray(const InBcd: TBcd; var nibbleArray: TNibbleArray): Integer;
  var
    Index, OutIndex, NotUsed: Integer;
    B: Byte;
    Started: Boolean;
  begin
    OutIndex := 0;
    Started := False;
    NotUsed := 0;
  // the number is 1 or greater
     for Index := 0 to 31 do
     begin
        B := Byte(InBcd.Fraction[Index]);
        if (Started) or (B > 0) then
        begin
          if Started or (Byte(B SHR 4) > 0) then
          begin
            nibbleArray[OutIndex] := Byte(B SHR 4);
            Inc(OutIndex);
            Started := True;
          end else
            Inc(NotUsed);
          if NotUsed + OutIndex >= InBcd.Precision then
            break;
          if Started or (Byte(Byte(B AND 15)) > 0) then
          begin
            nibbleArray[OutIndex] := Byte(B AND 15);
            Inc(OutIndex);
            Started := True;
          end;
        end else
          Inc(NotUsed, 2);
        if NotUsed + OutIndex >= InBcd.Precision then
          break;
     end;
     Result := OutIndex;
  end;
                              
var
  nibbleArray1, nibbleArray2: TNibbleArray;
  I, Digits, Digits1, Digits2: Integer;
begin
  Result := 0;
  Digits1 := LoadNibbleArray(Bcd1, nibbleArray1);
  Digits2 := LoadNibbleArray(Bcd2, nibbleArray2);
  Digits := Min(Digits1, Digits2);
  for I := 0 to Digits -1 do
  begin
    Result := nibbleArray1[I] - NibbleArray2[I];
    if Result <> 0 then
    begin
      if Result > 0 then
        Result := 1
      else
        Result := -1;
      Exit;
    end;
  end;
  for I := Digits to Digits1 -1 do
    if nibbleArray1[i] <> 0 then
      Result := 1;
  for I := Digits to Digits2 -1 do
    if nibbleArray2[i] <> 0 then
      Result := -1;
end;

{ Shift Fractions one Nibble to Left }
procedure MoveNibblesToLeft(var BcdVal: TBcd; BcdIn: TBcd);
var
  I, InIndex, OutIndex, InPrecision: Integer;
  B: Byte;
  nibbleArray: TNibbleArray;
begin
  InPrecision := BcdIn.Precision;
  OutIndex := 0;
  if (Byte(BcdIn.Fraction[0]) SHR 4) <> 0 then
    BcdError(sBcdOverflow);
  for InIndex := 0 to (InPrecision div 2)  do
  begin                                      
    B := Byte(BcdIn.Fraction[InIndex]);
    if InIndex > 0 then
    begin
      nibbleArray[OutIndex] := Byte(B SHR 4);
      inc(OutIndex);
    end;
    nibbleArray[OutIndex] := Byte(B AND 15);
    Inc(OutIndex);
  end;
  nibbleArray[OutIndex] := Byte(0);
  InIndex := 0;
  for I := 0 to (InPrecision div 2) do
  begin
    B := nibbleArray[InIndex] SHL 4;
    Inc(InIndex);
    B := B OR (nibbleArray[InIndex] AND 15);
    BcdVal.Fraction[I] := B;
    Inc(InIndex);
  end;
end;


function NormalizeBcd(const InBcd: TBcd; var OutBcd: TBcd; const Prec, Scale: Word): Boolean;
var
  I, InScale, Distance: Integer;
  InDecPos, OutDecPos: Integer;
  MoveLeft, Negative: Boolean;
  TempBcd: TBcd;
begin
  Result := True;
  if (Word(InBcd.Precision) = Prec) and (Word(InBcd.SignSpecialPlaces and 63) = Scale) then
    OutBcd := InBcd
  else
  begin
    OutBcd.Precision := Prec;
    OutBcd.SignSpecialPlaces := Scale;
    InScale := BcdScale(InBcd);
    InDecPos := InBcd.Precision - InScale;
    OutDecPos := Prec - Scale;
    if InDecPos > OutDecPos then
    begin
      Distance := InDecPos - OutDecPos;
      MoveLeft := True;
    end else
    begin
      Distance := OutDecPos - InDecPos ;
      MoveLeft := False;
    end;
    Negative := IsBcdNegative(InBcd);
    if Distance = 0 then
      OutBcd.Fraction := InBcd.Fraction
    else
    begin
      TempBcd := InBcd;
      for I := 0 to ((Prec div 2) -1) do
        OutBcd.Fraction[I] := Byte(0);
      if Prec mod 2 = 1 then
        OutBcd.Fraction[(Prec div 2)] := Byte(0);
      I := 0;
      while I < Distance do
      begin
        if MoveLeft then
          MoveNibblesToLeft(OutBcd, InBcd)
        else
        begin
          if Distance mod 2 = 0 then
          begin
            MoveBytesToRight(OutBcd, TempBcd);
            TempBcd := OutBcd;
            Inc(I);
          end else
          begin
            MoveNibblesToRight(OutBcd, TempBcd);
            TempBcd := OutBcd;
          end;
        end;
        Inc(I);
      end;
    end;
    if Negative then
      OutBcd.SignSpecialPlaces := (OutBcd.SignSpecialPlaces and 63) or (1 shl 7);
  end;
end;

function NumberOfDigits(const ABcd: TBcd): Integer;
var
  P: Word;
begin
  Result := ABcd.Precision;
  P := 0;
  while (Result > 1) and (Abcd.Fraction[P] = 0) do
  begin
    Dec(Result, 2);   // decrement two nibbles per byte
    Inc(P);
  end;
end;

function CompactBcd(const ABcd: TBcd; const MinSize: Integer): TBcd;
var
  I, J, CharsToMove: Integer;
begin
  if ABcd.Precision <= MinSize then
  begin
    Result := ABcd;
    exit;
  end;
  Result.Precision := ABcd.Precision;
  Result.SignSpecialPlaces := ABcd.SignSpecialPlaces;
  ZeroBcd(Result, True);
  I := 0;
  while (Result.Precision > MinSize) and (ABcd.Fraction[I] = 0) do
  begin
    Dec(Result.Precision,2);
    Inc(I);
  end;
  CharsToMove := (2+ (Result.Precision + BcdScale(Result))) div 2;
  if CharsToMove > SizeOf(Result.Fraction) then CharsToMove := SizeOf(Result.Fraction);
  for J := 0 to CharsToMove - 1 do
    Result.Fraction[J] := ABcd.Fraction[I + J];
end;

procedure NormalizeBcdPair(const BcdIn1, BcdIn2: TBcd; var bcdOut1, bcdOut2: TBcd; ExtraDigits: Word = 0 );
var
  MaxDigits, MaxScale: Word;
  Bcd1, Bcd2: TBcd;

  { Guarantee Bcd has even number Precision }
  function AdjustNibbles(ABcd: TBcd): TBcd;
  var
    I, Start: Integer;
  begin
    Result := ABcd;
    if (ABcd.Precision mod 2) <> 0 then
    begin
      Result.Fraction[0] := 0;
      Result.Precision := ABcd.Precision + 1;
      MoveNibblesToRight(Result, ABcd);
    end;
    { Guarantee unused Nibbles are blank}
    Start := (Result.Precision div 2);
    for I := Start to SizeOf(Result.Fraction) -1 do
      Result.Fraction[I] := 0;
  end;

begin
  Bcd1 := AdjustNibbles(BcdIn1);
  Bcd2 := AdjustNibbles(BcdIn2);
  if (Bcd1.Precision > 32) or (Bcd2.Precision > 32) then
  begin
    MaxDigits := Max(NumberOfDigits(bcdIn1), NumberOfDigits(bcdIn2));
    if MaxDigits < Bcd1.Precision then MaxDigits := Bcd1.Precision;
    if MaxDigits < Bcd2.Precision then MaxDigits := Bcd2.Precision;
    Bcd1 := CompactBcd(Bcd1, MaxDigits);
    Bcd2 := CompactBcd(Bcd2, MaxDigits);
  end;
  MaxDigits := Max(Bcd1.Precision, Bcd2.Precision);
  MaxScale := Max(BcdScale(Bcd1), BcdScale(Bcd2));
  { ensure that MaxDigits is large enough: for example, if Bcd1 is 6.0 and
    Bcd2 10.5, then MaxDigits needs to be 12, not 10 }
  while (MaxDigits < (SizeOf(Bcd1.Fraction) * 2)) and
        ((MaxDigits - MaxScale < (ExtraDigits + Bcd1.Precision) - BcdScale(Bcd1)) or
         (MaxDigits - MaxScale < (ExtraDigits + Bcd2.Precision) - BcdScale(Bcd2))) do
    Inc(MaxDigits, 2);
  NormalizeBcd(Bcd1, BcdOut1, MaxDigits, MaxScale);
  NormalizeBcd(Bcd2, BcdOut2, MaxDigits, MaxScale);
end;

function BcdCompare(const Bcd1, Bcd2: TBcd): Integer;
var
  Digits1,Digits2: ShortInt;
  Negative: Boolean;
begin
  if (Bcd1.SignSpecialPlaces and (1 shl 7)) <> (Bcd2.SignSpecialPlaces and (1 shl 7)) then
  begin  // if Negative setting doesn't match.
    if (Bcd1.SignSpecialPlaces and (1 shl 7)) <> 0 then
      Result := -1
    else
      Result := 1;
  end else
  begin  // both Negative or both Positive
    Negative := (Bcd1.SignSpecialPlaces and (1 shl 7)) <> 0;
    Digits1 := SignificantIntDigits(Bcd1, SmallInt(Bcd1.Precision - (Bcd1.SignSpecialPlaces and 63)));
    Digits2 := SignificantIntDigits(Bcd2, SmallInt(Bcd2.Precision - (Bcd2.SignSpecialPlaces and 63)));
    if Digits1 <> Digits2 then
    begin
      if Digits1 > Digits2 then
        Result := 1
      else
        Result := -1;
    end else
    begin
      Result := CompareSignificantFractions(Bcd1, Bcd2, Digits1);
    end;
    if Negative then Result := -Result;
  end;
end;

procedure BcdSubtract(const bcdIn1, bcdIn2: TBcd; var bcdOut: TBcd);
var
  newBcd2: TBcd;
begin
  newBcd2 := bcdIn2;
  newBcd2.SignSpecialPlaces := ReverseNegative(newBcd2.SignSpecialPlaces);
  BcdAdd(bcdIn1, newBcd2, bcdOut);
end;

procedure BcdMultiply(StringIn1, StringIn2: string; var bcdOut: TBcd); overload;
var
  Output: string;
  NegCount: Integer;
begin
  NegCount := 0;
  if StringIn1[1] = '-' then
  begin
    Inc(NegCount);
    StringIn1 := Copy(StringIn1, 2, Length(StringIn1)-1);
  end;
  if StringIn2[1] = '-' then
  begin
    Inc(NegCount);
    StringIn2 := Copy(StringIn2, 2, Length(StringIn2)-1);
  end;
  Output := _Multiply(StringIn1, StringIn2);
  bcdOut := StrToBcd(Output);
  if (NegCount mod 2) <> 0 then
    bcdOut.SignSpecialPlaces := (bcdOut.SignSpecialPlaces and 63) or (1 shl 7);
end;

procedure BcdMultiply(const bcdIn1, bcdIn2: TBcd; var bcdOut: TBcd);
begin
  BcdMultiply(BcdToStr(bcdIn1), BcdToStr(bcdIn2), bcdOut);
end;

procedure BcdMultiply(const bcdIn: TBcd; const DoubleIn: Double; var bcdOut: TBcd);
begin
  BcdMultiply(BcdToStr(bcdIn), FloatToStr(DoubleIn), bcdOut);
end;

procedure BcdMultiply(const bcdIn: TBcd; const StringIn: string; var bcdOut: TBcd);
begin
  BcdMultiply(BcdToStr(bcdIn), StringIn, bcdOut);
end;

procedure BcdDivide(Dividend, Divisor: string; var bcdOut: TBcd); overload;
var
  Output: string;
  NegCount: Integer;
  MaxDecimals: Byte;
begin
  if (Divisor = '0') or (Divisor = '') then
    BcdError(SDivByZero);
  NegCount := 0;
  MaxDecimals := (bcdOut.signSpecialPlaces and 63) + _DefaultDecimals;
  if Dividend[1] = '-' then
  begin
    Inc(NegCount);
    Dividend := Copy(Dividend, 2, Length(Dividend)-1);
  end;
  if Divisor[1] = '-' then
  begin
    Inc(NegCount);
    Divisor := Copy(Divisor, 2, Length(Divisor)-1);
  end;
  Output := RoundAt(_Divide(Dividend, Divisor), MaxDecimals);
  bcdOut := StrToBcd(Output);
  if (NegCount mod 2) <> 0 then
    bcdOut.SignSpecialPlaces := (bcdOut.SignSpecialPlaces and 63) or (1 shl 7);
end;

procedure BcdDivide(const Dividend, Divisor: TBcd; var bcdOut: TBcd);
begin
  BcdDivide(BcdToStr(Dividend), BcdToStr(Divisor), bcdOut);
end;

procedure BcdDivide(const Dividend: TBcd; const Divisor: Double; var bcdOut: TBcd);
begin
  BcdDivide(BcdToStr(Dividend), FloatToStr(Divisor), bcdOut);
end;

procedure BcdDivide(const Dividend: TBcd; const Divisor: string; var bcdOut: TBcd);
begin
  BcdDivide(BcdToStr(Dividend), Divisor, bcdOut);
end;

procedure BcdAdd(const bcdIn1, bcdIn2: TBcd; var bcdOut: TBcd);
var
  bcd1, bcd2: TBcd;
  Prec, Scale: Word;
  Neg1, Neg2: Boolean;
  Digits1, Digits2: Integer;
begin
  NormalizeBcdPair(bcdIn1, bcdIn2, bcd1, bcd2, 1);
  Prec := bcd1.Precision;
  Scale := bcd1.SignSpecialPlaces;
  Neg1 := (BcdIn1.SignSpecialPlaces and (1 shl 7)) <> 0;
  Neg2 := (BcdIn2.SignSpecialPlaces and (1 shl 7)) <> 0;
  bcdOut.Precision := Prec;
  if (Neg1 = Neg2) or ((bcd1.SignSpecialPlaces and 63) >= (bcd2.SignSpecialPlaces and 63)) then
    bcdOut.SignSpecialPlaces := Scale
  else
    bcdOut.SignSpecialPlaces := ReverseNegative(Scale);
  ZeroBcd(bcdOut, True);
  if Neg1 = Neg2 then
    AddNormalizedFractions(bcd1, bcd2, bcdOut)
  else
  begin
    Digits1 := SignificantIntDigits(bcd1, Prec);
    Digits2 := SignificantIntDigits(bcd2, Prec);
    if Digits1 > Digits2 then
      SubtractNormalizedFractions(bcd1, bcd2, bcdOut)
    else if Digits2 > Digits1 then
    begin
      SubtractNormalizedFractions(bcd2, bcd1, bcdOut);
      bcdOut.SignSpecialPlaces := ReverseNegative(bcdOut.SignSpecialPlaces);
    end
    else if CompareNormalizedFractions(bcd1, bcd2, Prec, Prec) >= 0 then
      SubtractNormalizedFractions(bcd1, bcd2, bcdOut)
    else
    begin
      SubtractNormalizedFractions(bcd2, bcd1, bcdOut);
      bcdOut.SignSpecialPlaces := ReverseNegative(bcdOut.SignSpecialPlaces);
    end;
  end;
end;

function BcdToStr(const Bcd: TBcd): string;
var
  I: Integer;
  DecPos: SmallInt;
  Negative: Boolean;
  C, Dot: Char;
begin
  if (Bcd.Precision = 0) or (Bcd.Precision > MaxFMTBcdFractionSize) or (BcdScale(Bcd) > Bcd.Precision) then
    OverFlowError(SBcdOverFlow)
  else
  begin
    Dot := DecimalSeparator;
    Negative := Bcd.SignSpecialPlaces and (1 shl 7) <> 0;
    DecPos := ShortInt(Bcd.Precision - (Bcd.SignSpecialPlaces and 63));
    Result := '';
    for I := 0 to Bcd.Precision - 1 do
    begin
      if I = DecPos then
      begin
        if I = 0 then
          Result := '0' + Dot
        else
          Result := Result + Dot;
      end;
      C := Char(GetBcdDigit(Bcd, I) + 48);
      { avoid leading 0's }
      if (Result <> '') or (C <> '0') or (I >= DecPos) then
        Result := Result + C;
    end;
    { if there is a decimal trim trailing '0's }
    if DecPos < Bcd.Precision then
    begin
      while Result[Length(Result)] = '0' do
        Result := Copy(Result, 1, Length(Result) - 1);
      if Result[Length(Result)] = Dot then
        Result := Copy(Result, 1, Length(Result) - 1);
    end;
    if Result = '' then
      Result := '0'
    else if Negative then
      Result := '-' + Result;
  end;
end;

function BcdPrecision(const Bcd: TBcd): Word;
begin
  Result := Bcd.Precision - BcdScale(Bcd);
end;

function BcdScale(const Bcd: TBcd): Word;
begin
  Result := (Bcd.SignSpecialPlaces and 63);
end;

function IsBcdNegative(const Bcd: TBcd): Boolean;
begin
  Result := (Bcd.SignSpecialPlaces and (1 shl 7)) <> 0;
end;

function IsBcdZero(const Bcd: TBcd): Boolean;
var
  I, Scale: Integer;
begin
  Result := True;
  I := 0;
  Scale := BcdScale(Bcd);
  while Result and (I < Scale div 2) do
  begin
    if Byte(Bcd.Fraction[I]) <> 0 then
      Result := False;
    Inc(I);
  end;
{ if odd nibble, check it }
  if Result and (Scale mod 2 > 0) then
    if (Byte(Bcd.Fraction[I]) SHR 4) > 0 then
      Result := False;
end;











function CurrToBCD(const Curr: Currency; var BCD: TBcd; Precision: Integer = 32;
  Decimals: Integer = 4): Boolean;
var
  Temp: Currency;
  BcdIndex, StrIndex, StartPos, Digits: Integer;
  B1, B2, DotPos: Byte;
  BcdStr: string;
  Dot: Char;

  function GetNextByte(): Byte;
  begin
    Result := 0;
    if BcdIndex < StartPos then
       Exit;
    if BcdStr[StrIndex] = Dot then
      Inc(StrIndex);
    if StrIndex <= Digits then
    begin
      Result := Byte(BcdStr[StrIndex]) - 48;
      Inc(StrIndex);
    end;
  end;

begin
  Dot := DecimalSeparator;
  Bcd.Precision := Precision;
  Bcd.SignSpecialPlaces := Decimals;
  for BcdIndex := 0 to 31 do
    Bcd.Fraction[BcdIndex] := 0;
  if Curr = 0 then
  begin
    Result := True;
    Exit;
  end;
  if Curr < 0 then
    Temp := -Curr
  else
    Temp := Curr;
  BcdStr := CurrToStr(Temp);
  Digits := Length(BcdStr);
  DotPos := Pos(Dot, BcdStr);
  if DotPos > 0 then
    StartPos := Precision - ((DotPos - 1) + Decimals)
  else
    StartPos := Precision - (Digits + Decimals);
  StrIndex := 1;
  BcdIndex := 0;
  while BcdIndex < Precision do
  begin
    B1 := GetNextByte();
    Inc(BcdIndex);
    B2 := GetNextByte();
    PutTwoBcdDigits(B1, B2, Bcd, BcdIndex-1);
    Inc(BcdIndex);
  end;
  if Curr < 0 then
    Bcd.SignSpecialPlaces := (Bcd.SignSpecialPlaces and 63) or (1 shl 7);
  Result := True;
end;

function BCDToCurr(const BCD: TBcd; var Curr: Currency): Boolean;
var
  Scale, I: Integer;
  Negative: Boolean;
  b: Byte;
begin
  Curr := 0;
  Negative := (Bcd.SignSpecialPlaces and (1 shl 7)) <> 0;
  Scale := (Bcd.SignSpecialPlaces and 63);
  for I := 0 to Bcd.Precision -1 do
  begin
    b := GetBcdDigit(Bcd, I);
    if b <> 0 then
      Curr := Curr + PutCurrencyDigit(b, Bcd.Precision - (Scale + I));
  end;
  if Scale > 4 then
  begin  { 0.12345 = 0.1234, but 0.123450000001 is rounded up to 0.1235 }
    b := GetBcdDigit(Bcd, 4 + (Bcd.Precision - Scale));
    if b >= 5 then
      if b > 5 then
        Curr := Curr + 0.0001
      else
        for I := 5 + (Bcd.Precision - Scale)  to Bcd.Precision -1 do
          if GetBcdDigit(Bcd, I) <> 0 then
          begin
            Curr := Curr + 0.0001;
            break;
          end;
  end;
  if Negative then
    Curr := -Curr;
  Result := True;
end;

{ ffGeneral - General number format. The value is converted to the shortest
  possible decimal string using fixed format. Trailing zeros
  are removed from the resulting string, and a decimal point appears only
  if necessary. The resulting string uses fixed point format if the number
  of digits to the left of the decimal point in the value is less than or
  equal to the specified precision.  Otherwise an exception is thrown }

function BcdGeneralFormat(const Bcd: TBcd; const Precision, Digits: Integer): string;
begin
  Result := BcdToStr(Bcd);
end;
  
{ ffExponent - Scientific format. Not supported for FMTBcd -- Bcd is 
  by definition fixed format }

function BcdScientificFormat(const Bcd: TBcd; const Precision, Digits: Integer): string;
begin
  BcdError(SInvalidFormatType);
end;

{ ffFixed - Fixed point format. The value is converted to a string of the
  form "-ddd.ddd...". The resulting string starts with a minus sign if the
  number is negative, and at least one digit always precedes the decimal
  point. The number of digits after the decimal point is given by the Digits
  parameter--it must be between 0 and 18. If the value has more decimal values
  than permitted by Digits, it is truncated.  If the number of digits to the
  left of the decimal point is greater than the specified precision, an
  exception is thrown
  ffNumber - Number format. The ffNumber format corresponds to the ffFixed
  format, except that the resulting string contains thousand separators. }

function BcdFixedOrNumberFormat(const Start: string; Format: TFloatFormat; const Precision, Digits: Integer): string;
var
  P, DecPos, DecLen, Len: Integer;
  AddCommaDigits, DecDigits, BufPos: Integer;
  Dot, Comma: Char;
begin
  Comma := ThousandSeparator;
  Dot := DecimalSeparator;
  BufPos := 0;
  Result := Start;
  Len := Length(Start);
  P := 1; // position in source string
  DecDigits := -1;
  DecPos := Pos(Dot, Result);
  DecLen := Length(Dot);
  if Format = ffNumber then
  begin
    AddCommaDigits := DecPos;
    if AddCommaDigits > 0 then
      Dec(AddCommaDigits)
    else
      AddCommaDigits := Length(Result);
  end else
    AddCommaDigits := 0;
  if Start[1] = '-' then
  begin
    Inc(BufPos); // current char is ok
    Inc(P);
    if AddCommaDigits > 0 then
      Dec(AddCommaDigits);
  end;
  while P <= Len do
  begin
    Inc(BufPos); // current char is ok
    if (P >= DecPos) and (P < DecPos + DecLen) then
      DecDigits := 0
    else if DecDigits > -1 then
      Inc(DecDigits);
    Inc(P);
    if AddCommaDigits > 3 then
    begin
      Dec(AddCommaDigits);
      if (AddCommaDigits mod 3 = 0) then
      begin
        Result := Copy(Result, 1, BufPos) +  Comma + Copy(Result, BufPos + 1, MaxInt);
        Inc(BufPos, Length(Comma));
      end;
    end;
  end;
  if DecDigits = -1 then
  begin
    if Digits > 0 then
    begin
      Result := Copy(Result, 1, BufPos) +  Dot + Copy(Result, BufPos + 1, MaxInt);
      Inc(BufPos, DecLen);
    end;
    DecDigits := 0;
  end;
  if DecDigits < Digits then
    Result := Result + StringOfChar('0', Digits - DecDigits);
  if Pos(Dot, Result) = BufPos + 1 - DecLen then
    SetLength(Result, BufPos - DecLen);
  Result := RoundAt( Result, Digits);
end;

{ ffCurrency - Currency format. The value is converted to a string that
  represents a currency amount. The conversion is controlled by the
  CurrencyString, CurrencyFormat, NegCurrFormat, ThousandSeparator, and
  DecimalSeparator global variables
  The number of digits after the decimal point is given by the Digits
  parameter--it must be between 0 and 18. }

{  0 = '$1',   1 = '1$',    2 = '$ 1',  3 = '1 $' }
function AddCurrencySymbol(const Value, CurrSymbol: string; const CurrFormat: Byte): string;
begin
  case CurrFormat of 
    0: Result := Format('%s%s', [CurrSymbol, Value]);
    1: Result := Format('%s%s', [Value, CurrSymbol]);
    2: Result := Format('%s %s', [CurrSymbol, Value]);
    3: Result := Format('%s %s', [Value, CurrSymbol]);
  end;
end;

{   0 = '($1)'      4 = '(1$)'      8 = '-1 $'      12 = '$ -1'
    1 = '-$1'       5 = '-1$'       9 = '-$ 1'      13 = '1- $'
    2 = '$-1'       6 = '1-$'      10 = '1 $-'      14 = '($ 1)'
    3 = '$1-'       7 = '1$-'      11 = '$ 1-'      15 = '(1 $)'  }
function AddNegCurrencySymbol(const Value, CurrSymbol: string; const CurrFormat: Byte): string;
begin
  case CurrFormat of 
    0: Result := Format('(%s%s)', [CurrSymbol, Value]);
    1: Result := Format('-%s%s', [CurrSymbol, Value]);
    2: Result := Format('%s-%s', [CurrSymbol, Value]);
    3: Result := Format('%s%s-', [CurrSymbol, Value]);
    4: Result := Format('(%s%s)', [Value, CurrSymbol]);
    5: Result := Format('-%s%s', [Value, CurrSymbol]);
    6: Result := Format('%s-%s', [Value, CurrSymbol]);
    7: Result := Format('%s%s-', [Value, CurrSymbol]);
    8: Result := Format('-%s %s', [Value, CurrSymbol]);
    9: Result := Format('-%s %s', [CurrSymbol, Value]);
   10: Result := Format('%s %s-', [Value, CurrSymbol]);
   11: Result := Format('%s %s-', [CurrSymbol, Value]);
   12: Result := Format('%s %s', [CurrSymbol, Value]);
   13: Result := Format('%s -%s', [Value, CurrSymbol]);
   14: Result := Format('(%s- %s)', [CurrSymbol, Value]);
   15: Result := Format('(%s %s)', [Value, CurrSymbol]);
  end;
end;

function BcdCurrencyFormat(const Bcd: TBcd; const Start: string; const Precision, Digits: Integer): string;
begin
  Result := BcdFixedOrNumberFormat(Start, ffNumber, Precision, Digits);
  if IsBcdNegative(Bcd) then
    Result := AddNegCurrencySymbol(Copy(Result, 2, Length(Result)),
      CurrencyString, NegCurrFormat)
  else
    Result := AddCurrencySymbol(Result, CurrencyString, CurrencyFormat);
end;

function BcdToStrF(const Bcd: TBcd; Format: TFloatFormat; const Precision, Digits: Integer): string;
begin
  Result := BcdGeneralFormat(Bcd, Precision, Digits);
  case Format of
    ffExponent:
      Result := BcdScientificFormat(Bcd, Precision, Digits);
    ffCurrency:
      Result := BcdCurrencyFormat(Bcd, Result, Precision, Digits);
    ffFixed, ffNumber:
      Result := BcdFixedOrNumberFormat(Result, Format, Precision, Digits);
  end;
end;

function ExtractMoneySymbol( var CurrencyPos: Integer; var HoldFormat: String): string;
var
  TempPos: Integer;
begin
  TempPos := CurrencyPos;
  Result := '$';
  while (CurrencyPos > 0) and (HoldFormat[CurrencyPos-1] = ' ') do
  begin
    Dec(CurrencyPos);
    Result := ' ' + Result;
  end;
  while (TempPos < Length(HoldFormat)) and (HoldFormat[TempPos+1] = ' ') do
  begin
    Inc(TempPos);
    Result := Result + ' ';
  end;
  HoldFormat := StringReplace(HoldFormat, Result, '', []);
  Result := StringReplace(Result, '$', CurrencyString, []);
end;

procedure CalcPrecisionAndDigits( const HoldFormat: string; var Precision, Digits, ReqIntDigits: Integer);
begin
  if Digits > 0 then
  begin
    ReqIntDigits := Digits -1;
    Precision := Length(HoldFormat) -1;
    Digits := Length(HoldFormat) - Digits;
  end else
  begin
    Precision := Length(HoldFormat);
    ReqIntDigits := Precision;
    Digits := 0;
  end;
end;

function PadInputString(var InputString: string; var Precision: Integer; const ReqIntDigits: Integer): Boolean;
var
  DecSep: Char;
  IntDigits, PadCount: Integer;
begin
  Result := True;
  DecSep := DecimalSeparator;
  IntDigits := Pos(DecSep, InputString);
  if IntDigits = 0 then
    IntDigits := Length(InputString)
  else
    Dec(IntDigits);
  PadCount := ReqIntDigits - IntDigits;
  if PadCount > 0 then
  begin
    InputString := '1' + StringOfChar('0', PadCount -1) + InputString;
    Inc(Precision);
  end else
    Result := False;
end;

procedure AddCurrency(var Output: string; const CurrencyPos: Integer; const MoneySymbol: string);
begin
  if CurrencyPos = 1 then
    Output := MoneySymbol + Output
  else if CurrencyPos >= Length(Output) then
    Output := Output + MoneySymbol
  else
    Output := Copy(Output, 1, CurrencyPos -1 ) + MoneySymbol +
      Copy(Output, CurrencyPos, Length(Output));
end;

type
  TLiteralInfo = class
  private
    FValue: string;
    FPosition: Integer;
  public
    property Value: string read FValue write FValue;
    property Position: Integer read FPosition write FPosition;
    constructor Create(Val: string; Pos: Integer);
  end;

constructor TLiteralInfo.Create;
begin
  FValue := Val;
  FPosition := Pos;
end;

procedure ExtractLiterals(LiteralList: TList; var HoldFormat: string);
const
  FormatChars = ['#', '0', ',', '.'];
  LiteralMarkers = [#39, '"'];
var
  PStart: Integer;
  C: Char;
  I, Pos, LiteralPos: Integer;
  LiteralString: string;
  InLiteral: Boolean;
begin
  InLiteral := False;
  LiteralPos := 0;
  Pos := 1;
  LiteralString := '';
  while Pos <= Length(HoldFormat) do
  begin
    C := HoldFormat[Pos];
    if C in LiteralMarkers then
      InLiteral := not InLiteral
    else if not (C in FormatChars) then
    begin
      LiteralPos := Pos - 1;
      PStart := Pos;
      while (InLiteral) or not (HoldFormat[Pos] in FormatChars) do
      begin
        Inc(Pos);
        if (Pos >= Length(HoldFormat)) or (HoldFormat[Pos] in LiteralMarkers) then
        begin
          InLiteral := False;
          break;
        end;
      end;
      LiteralString := Copy(HoldFormat, PStart, Pos-PStart);
    end;
    if (not InLiteral) and (LiteralString <> '') then
    begin
      LiteralList.Add(TLiteralInfo.Create(LiteralString, LiteralPos));
      LiteralString := '';
    end;
    Inc(Pos);
  end;
  for I := 0 to LiteralList.Count -1 do
    HoldFormat := StringReplace(HoldFormat, TLiteralInfo(LiteralList.Items[I]).Value, '', []);
end;

procedure AddLiterals(LiteralList: TList; var Output: string; const Format: string);
var
  I, Add: Integer;
  Pos: Integer;
begin
  Add := (Length(Output)+1) - Length(Format);
  if Add < 0 then Add := 0;
  for I := 0 to LiteralList.Count -1 do
  begin
    Pos := TLiteralInfo(LiteralList.Items[I]).Position;
    if Pos = 0 then
      Output := TLiteralInfo(LiteralList.Items[I]).Value + Output
    else if (Pos + Add) < Length(Output) then
    begin
      Inc(Pos, Add);
      Output := Copy(Output, 1, Pos -1 ) + TLiteralInfo(LiteralList.Items[I]).Value +
           Copy(Output, Pos, length(Output) - (Pos -1));
    end else
      Output := Output + TLiteralInfo(LiteralList.Items[I]).Value;
    TLiteralInfo(LiteralList.Items[I]).Free;
  end;
end;

function FormatOneBcd(const Format: string; Bcd: TBcd): string;
const
  Dot: Char = '.';
var
  Precision, Digits, ReqDigits: Integer;
  TempPos, CurrencyPos: Integer;
  MoneySymbol: string;
  FFormat: TFloatFormat;
  LeftPadZeros, RightPadZeros, UseCommas: Boolean;
  Neg, HoldFormat: string;
  LiteralList: TList;
begin
  Neg := '';
  LiteralList := TList.Create;
  try
    HoldFormat := Format;
    CurrencyPos := Pos('$', Format);
    if CurrencyPos > 0 then
      MoneySymbol := ExtractMoneySymbol(CurrencyPos, HoldFormat);
    ExtractLiterals(LiteralList,HoldFormat);
    UseCommas := (Pos(',', HoldFormat) > 0);
    if UseCommas then
      HoldFormat := StringReplace(HoldFormat, ',', '', []);
    Digits := Pos(Dot, HoldFormat);
    CalcPrecisionAndDigits(HoldFormat, Precision, Digits, ReqDigits);
    TempPos := Pos('0', HoldFormat);
    LeftPadZeros := (TempPos > 0) and (TempPos < ReqDigits);
    TempPos := Pos('0', Copy(HoldFormat, ReqDigits, Digits + 1));
    RightPadZeros := TempPos > 0;
    if UseCommas then
      FFormat := ffNumber
    else
      if RightPadZeros or (Digits < BcdScale(Bcd)) then
        FFormat := ffFixed
      else
        FFormat := ffGeneral;
    Result := BcdGeneralFormat(Bcd, Precision, Digits);
    if IsBcdNegative(Bcd) then
    begin
      Neg := '-';
      Result := StringReplace(Result, Neg, '', []);
      Inc(CurrencyPos);
    end;
    if LeftPadZeros then
      LeftPadZeros := PadInputString(Result, Precision, ReqDigits);
    if FFormat <> ffGeneral then
      Result := BcdFixedOrNumberFormat(Result, FFormat, Precision, Digits);
    if LeftPadZeros then
      Result := Neg + '0' + Copy(Result, 2, Length(Result) -1)
    else
      Result := Neg + Result;
    if MoneySymbol <> '' then
      AddCurrency(Result, CurrencyPos, MoneySymbol);
    AddLiterals(LiteralList, Result, HoldFormat);
  finally
    LiteralList.Free;
  end;      
end;

type
  TFormatSection = (sectionPositive, sectionNegative, sectionZero);

function GetFormat(const Format: string; Section: TFormatSection): string;
const
  Separator: Char = ';';
  LiteralMarkers = [#39, '"'];
var
  InLiteral: Boolean;
  P, PThisSection, SectionCount, Len: Integer;
  SPos: string;
begin
  if Pos(Separator,Format) = 0 then
    Result := Format
  else
  begin
    PThisSection := 1;
    SectionCount := 0;
    P := PThisSection;
    InLiteral := False;
    Len := Length(Format);
    while P <= Len do
    begin
      if (Format[P] in LiteralMarkers) then
        InLiteral := not InLiteral
      else if (Format[P] = Separator) and (not InLiteral) then
      begin
        if Ord(Section) = SectionCount then
          Break
        else if SectionCount = 0 then // remember first section
          SPos := Copy(Format, PThisSection, P - PThisSection);
        PThisSection := P + 1;
        Inc(SectionCount);
      end;
      Inc(P);
    end;
    // if we get here, we reached the end of the format string
    if SectionCount = Ord(Section) then
      Result := Copy(Format, PThisSection, P - PThisSection)
    else
      Result := SPos;
  end;
end;

function FormatBcd(const Format: string; Bcd: TBcd): string;
begin
  if IsBcdNegative(Bcd) then
    Result := FormatOneBcd(GetFormat(Format, sectionNegative), Bcd)
  else if IsBcdZero(Bcd) then
    Result := FormatOneBcd(GetFormat(Format, sectionZero), Bcd)
  else
    Result := FormatOneBcd(GetFormat(Format, sectionPositive), Bcd);
end;

function OldCurrToBCD(const Curr: Currency; var BCD: TBcd; Precision: Integer = 32;
  Decimals: Integer = 4): Boolean;
const
  Power10: array[0..3] of Single = (10000, 1000, 100, 10);
var
  Digits: array[0..63] of Byte;
asm
        PUSH    EBX
        PUSH    ESI
        PUSH    EDI
        MOV     ESI,EAX
        XCHG    ECX,EDX








        MOV     [ESI].TBcd.Precision,CL
        MOV     [ESI].TBcd.SignSpecialPlaces,DL
@@1:    SUB     EDX,4
        JE      @@3
        JA      @@2
        FILD    Curr  



        FDIV    Power10.Single[EDX*4+16]

        FISTP   Curr
        JMP     @@3
@@2:    DEC     ECX
        MOV     Digits.Byte[ECX],0
        DEC     EDX
        JNE     @@2
@@3:    MOV     EAX,Curr.Integer[0]
        MOV     EBX,Curr.Integer[4]
        OR      EBX,EBX
        JNS     @@4
        NEG     EBX
        NEG     EAX
        SBB     EBX,0
        OR      [ESI].TBcd.SignSpecialPlaces,80H
@@4:    MOV     EDI,10
@@5:    MOV     EDX,EAX
        OR      EDX,EBX
        JE      @@7
        XOR     EDX,EDX
        OR      EBX,EBX
        JE      @@6
        XCHG    EAX,EBX
        DIV     EDI
        XCHG    EAX,EBX
@@6:    DIV     EDI
@@7:    MOV     Digits.Byte[ECX-1],DL
        DEC     ECX
        JNE     @@5
        OR      EAX,EBX
        MOV     AL,0
        JNE     @@9
        MOV     CL,[ESI].TBcd.Precision
        INC     ECX
        SHR     ECX,1
@@8:    MOV     AX,Digits.Word[ECX*2-2]
        SHL     AL,4
        OR      AL,AH
        MOV     [ESI].TBcd.Fraction.Byte[ECX-1],AL
        DEC     ECX
        JNE     @@8
        MOV     AL,1
@@9:    POP     EDI
        POP     ESI
        POP     EBX
end;

function OldBCDToCurr(const BCD: TBcd; var Curr: Currency): Boolean;
const
  FConst10: Single = 10;
  CWNear: Word = $133F;
var
  CtrlWord: Word;
  Temp: Integer;
  Digits: array[0..63] of Byte;
asm
        PUSH    EBX
        PUSH    EDI
        PUSH    ESI










        XOR     EBX,EBX

        MOV     EDI,EAX
        MOV     ESI,EDX
        MOV     AL,0
        MOVZX   EDX,[EDI].TBcd.Precision
        OR      EDX,EDX
        JE      @@8
        LEA     ECX,[EDX+1]
        SHR     ECX,1
@@1:    MOV     AL,[EDI].TBcd.Fraction.Byte[ECX-1]
        MOV     AH,AL
        SHR     AL,4
        AND     AH,0FH
        MOV     Digits.Word[ECX*2-2],AX
        DEC     ECX
        JNE     @@1
        XOR     EAX,EAX
@@2:    MOV     AL,Digits.Byte[ECX]
        OR      AL,AL
        JNE     @@3
        INC     ECX
        CMP     ECX,EDX
        JNE     @@2
        FLDZ
        JMP     @@7
@@3:    MOV     Temp,EAX
        FILD    Temp
@@4:    INC     ECX
        CMP     ECX,EDX
        JE      @@5
        FMUL    [EBX].FConst10
        MOV     AL,Digits.Byte[ECX]
        MOV     Temp,EAX
        FIADD   Temp
        JMP     @@4
@@5:    MOV     AL,[EDI].TBcd.SignSpecialPlaces
        OR      AL,AL
        JNS     @@6
        FCHS
@@6:    AND     EAX,3FH
        SUB     EAX,4
        NEG     EAX
        CALL    FPower10
@@7:    FSTCW   CtrlWord
        FLDCW   [EBX].CWNear
        FISTP   [ESI].Currency
        FSTSW   AX
        NOT     AL
        AND     AL,1
        FCLEX
        FLDCW   CtrlWord
        FWAIT
@@8:
        POP     ESI
        POP     EDI
        POP     EBX
end;


initialization
  FMTBcdVariantType := TFMTBcdVariantType.Create;
finalization
  FreeAndNil(FMTBcdVariantType);
end.


